/*
 * Decompiled with CFR 0.152.
 */
package org.dynmap.s3lite.core.auth;

import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.time.temporal.TemporalAccessor;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.TreeMap;
import java.util.function.BinaryOperator;
import java.util.function.Function;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.dynmap.s3lite.api.auth.AwsCredentials;
import org.dynmap.s3lite.api.region.Region;
import org.dynmap.s3lite.core.auth.RegionAwareSigner;
import org.dynmap.s3lite.core.auth.SignableRequest;
import org.dynmap.s3lite.http.spi.SdkHttpUtils;
import org.dynmap.s3lite.http.spi.request.ImmutableRequest;
import org.dynmap.s3lite.http.spi.request.RequestBody;
import org.dynmap.s3lite.util.DigestUtils;
import org.dynmap.s3lite.util.StringUtils;

final class Aws4Signer
implements RegionAwareSigner {
    private static final Logger log = Logger.getLogger(Aws4Signer.class.getName());
    private static final String SERVICE_SIGNING_NAME = "s3";
    private static final String NO_PAYLOAD_HASH = DigestUtils.sha256Hex("");
    private static final Locale LOCALE_ENGLISH = Locale.ENGLISH;
    private static final DateTimeFormatter DATE_ONLY = DateTimeFormatter.ofPattern("yyyyMMdd").withZone(ZoneOffset.UTC);
    private static final DateTimeFormatter ISO_8601 = DateTimeFormatter.ofPattern("yyyyMMdd'T'HHmmss'Z'").withZone(ZoneOffset.UTC);
    private Region region;

    static Aws4Signer create() {
        return new Aws4Signer();
    }

    private Aws4Signer() {
    }

    @Override
    public void sign(SignableRequest request, AwsCredentials credentials) {
        Instant instant = Instant.now();
        String payloadHash = Aws4Signer.calculatePayloadHash(request);
        request.addHeader("Host", request.getEndpoint().getHost());
        request.addHeader("X-Amz-Date", ISO_8601.format(instant));
        request.addHeader("X-Amz-Content-Sha256", payloadHash);
        String signedHeaders = Aws4Signer.getSignedHeadersString(request.getHeaders().keySet());
        String scope = Aws4Signer.getScope(instant, this.region.getRegionName());
        String canonicalRequest = Aws4Signer.createCanonicalRequest(request, signedHeaders, payloadHash);
        String stringToSign = Aws4Signer.createStringToSign(canonicalRequest, scope, instant);
        String signature = Aws4Signer.calculateSignature(stringToSign, credentials, this.region.getRegionName(), instant);
        String authHeader = Aws4Signer.buildAuthorizationHeaders(signedHeaders, signature, credentials, scope);
        request.addHeader("Authorization", authHeader);
    }

    @Override
    public Region getRegion() {
        return this.region;
    }

    @Override
    public void setRegion(Region region) {
        this.region = region;
    }

    private static String calculatePayloadHash(ImmutableRequest request) {
        return request.getRequestBody().map(RequestBody::getContentStreamProvider).map(supplier -> {
            try (InputStream data = (InputStream)supplier.get();){
                String string = DigestUtils.sha256Hex(data);
                return string;
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }).orElse(NO_PAYLOAD_HASH);
    }

    private static String getScope(TemporalAccessor temporal, String regionName) {
        return DATE_ONLY.format(temporal) + "/" + regionName + "/" + SERVICE_SIGNING_NAME + "/" + "aws4_request";
    }

    private static String createCanonicalRequest(ImmutableRequest request, String signedHeaders, String payloadHash) {
        String canonicalRequest = request.getHttpMethod().name() + "\n" + Aws4Signer.getCanonicalizedResourcePath(request.getResourcePath()) + "\n" + Aws4Signer.getCanonicalizedQueryString(request.getParameters()) + "\n" + Aws4Signer.getCanonicalizedHeaderString(request.getHeaders()) + "\n" + signedHeaders + "\n" + payloadHash;
        log.config(() -> "AWS4 Canonical Request: " + canonicalRequest);
        return canonicalRequest;
    }

    private static String createStringToSign(String canonicalRequest, String scope, TemporalAccessor temporal) {
        String stringToSign = "AWS4-HMAC-SHA256\n" + ISO_8601.format(temporal) + "\n" + scope + "\n" + DigestUtils.sha256Hex(canonicalRequest);
        log.config(() -> "AWS4 String to Sign: " + stringToSign);
        return stringToSign;
    }

    private static String calculateSignature(String stringToSign, AwsCredentials credentials, String regionName, TemporalAccessor temporal) {
        byte[] kSecret = ("AWS4" + credentials.getAWSSecretKey()).getBytes(StandardCharsets.UTF_8);
        byte[] kDate = DigestUtils.hmacSha256(kSecret, DATE_ONLY.format(temporal));
        byte[] kRegion = DigestUtils.hmacSha256(kDate, regionName);
        byte[] kService = DigestUtils.hmacSha256(kRegion, SERVICE_SIGNING_NAME);
        byte[] kSigning = DigestUtils.hmacSha256(kService, "aws4_request");
        byte[] signature = DigestUtils.hmacSha256(kSigning, stringToSign);
        return DigestUtils.encodeHexString(signature);
    }

    private static String buildAuthorizationHeaders(String signedHeaders, String signature, AwsCredentials credentials, String scope) {
        String credential = "Credential=" + credentials.getAWSAccessKeyId() + "/" + scope;
        String headers = "SignedHeaders=" + signedHeaders;
        String sign = "Signature=" + signature;
        return "AWS4-HMAC-SHA256 " + credential + ", " + headers + ", " + sign;
    }

    private static String getCanonicalizedResourcePath(String resourcePath) {
        return SdkHttpUtils.urlEncodeIgnoreSlashes(resourcePath);
    }

    private static String getCanonicalizedQueryString(Map<String, List<String>> parameters) {
        Function<Map.Entry, String> keyMapper = Map.Entry::getKey;
        Function<Map.Entry, List> valueMapper = entry -> ((List)entry.getValue()).stream().sorted().collect(Collectors.toList());
        Map sortedParameters = parameters.entrySet().stream().collect(Collectors.toMap(keyMapper, valueMapper, Aws4Signer.throwingMerger(), TreeMap::new));
        return SdkHttpUtils.toQueryString(sortedParameters);
    }

    private static String getCanonicalizedHeaderString(Map<String, List<String>> headers) {
        return headers.entrySet().stream().sorted(Comparator.comparing(entry -> ((String)entry.getKey()).toLowerCase(LOCALE_ENGLISH))).map(entry -> {
            String headerName = ((String)entry.getKey()).toLowerCase(LOCALE_ENGLISH);
            String headerValues = ((List)entry.getValue()).stream().map(StringUtils::trimAll).collect(Collectors.joining(","));
            return headerName + ":" + headerValues + "\n";
        }).collect(Collectors.joining());
    }

    private static String getSignedHeadersString(Collection<String> headers) {
        return headers.stream().sorted(String.CASE_INSENSITIVE_ORDER).map(header -> header.toLowerCase(LOCALE_ENGLISH)).collect(Collectors.joining(";"));
    }

    private static <T> BinaryOperator<T> throwingMerger() {
        return (u, v) -> {
            throw new IllegalStateException("Duplicate key " + u);
        };
    }
}

